/*
 * 2D image rotation test case for GDMAC driver.
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/dma-mapping.h>
#include <linux/dmw96dma.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/wait.h>

static struct dmw96dma_image *alloc_image(int width, int height,
	enum dmw96dma_width fmt, int coherent)
{
	struct dmw96dma_image *img = kzalloc(sizeof(struct dmw96dma_image), GFP_KERNEL);
	int size = width * height * (1 << fmt);
	u32 *b, i = 0x01020304;

	img->format = fmt;
	img->coherent = coherent;
	img->width = width;
	img->height = height;

	if (coherent) {
		b = img->data.buf = dma_alloc_coherent(NULL, size, &img->data.coherent, GFP_KERNEL);
	} else {
		b = img->data.buf = kmalloc(size, GFP_KERNEL);
	}

	while (size > 0) {
		*b++ = i;
		i += 0x01010101;
		size -= 4;
	}

	return img;
}

static void free_image(struct dmw96dma_image *img)
{
	int size = img->width * img->height * (1 << img->format);

	if (img->coherent)
		dma_free_coherent(NULL, size, img->data.buf, img->data.coherent);
	else
		kfree(img->data.buf);

	kfree(img);
}

static int cmp_pixel_8(u8 *src, u8 *dst, int ws, int xs, int ys, int wd, int xd, int yd)
{
	return src[ys * ws + xs] == dst[yd * wd + xd];
}

static int cmp_pixel_16(u16 *src, u16 *dst, int ws, int xs, int ys, int wd, int xd, int yd)
{
	return src[ys * ws + xs] == dst[yd * wd + xd];
}

static int cmp_pixel_32(u32 *src, u32 *dst, int ws, int xs, int ys, int wd, int xd, int yd)
{
	return src[ys * ws + xs] == dst[yd * wd + xd];
}

static int cmp_image(struct dmw96dma_image *src, struct dmw96dma_image *dst,
	struct dmw96dma_rect *src_rect, struct dmw96dma_rect *dst_rect,
	enum dmw96dma_rotate degree)
{
	int ws = src->width, wd = dst->width;
	int x, y;
	enum dmw96dma_width fmt = src->format;
	void *bs = src->data.buf;
	void *bd = dst->data.buf;

	for (y = 0; y < src_rect->height; y++) {
		for (x = 0; x < src_rect->width; x++) {
			int xd, yd, cmp = 0;
			int xs = src_rect->x + x;
			int ys = src_rect->y + y;

			switch (degree) {
				case DMW96DMA_ROTATE_90_CW:
					xd = src_rect->height - y - 1;
					yd = x;
					break;
				case DMW96DMA_ROTATE_90_CCW:
					xd = y;
					yd = src_rect->width - x - 1;
					break;
				case DMW96DMA_ROTATE_180_CW:
					xd = src_rect->width - x - 1;
					yd = src_rect->height - y - 1;
					break;
				default:
					xd = x;
					yd = y;
			}
			xd += dst_rect->x;
			yd += dst_rect->y;

			switch (fmt) {
				case DMW96DMA_WIDTH_8:
					cmp = cmp_pixel_8(bs, bd, ws, xs, ys, wd, xd, yd);
					break;
				case DMW96DMA_WIDTH_16:
					cmp = cmp_pixel_16(bs, bd, ws, xs, ys, wd, xd, yd);
					break;
				case DMW96DMA_WIDTH_32:
					cmp = cmp_pixel_32(bs, bd, ws, xs, ys, wd, xd, yd);
					break;
			}

			if (!cmp) {
				printk("failed@x:%d,y:%d ", xs, ys);
				return -EIO;
			}
		}
	}

	return 0;
}


static volatile int rot_done;

struct rot_ctx {
	struct dmw96dma_list *list;
	struct dmw96dma_image *src;
	struct dmw96dma_image *dst;
	struct dmw96dma_rect *src_rect;
	struct dmw96dma_rect *dst_rect;
	enum dmw96dma_rotate degree;
	char *msg;

	wait_queue_head_t *wq;
};

static void rotate_image_finish(int status, void *context)
{
	struct rot_ctx *ctx = (struct rot_ctx *)context;
	struct dmw96dma_rect sr;
	struct dmw96dma_rect dr;

	if (status)
		printk("DMA rotate failed: %d\n", status);

	/* Very bad style to do this in tasklet context but its just a test... */
	sr.x = 0; sr.y = 0; sr.width = ctx->src->width; sr.height = ctx->src->height;
	dr.x = 0; dr.y = 0; dr.width = ctx->dst->width; dr.height = ctx->dst->height;
	if (cmp_image(ctx->src, ctx->dst, ctx->src_rect ? ctx->src_rect : &sr,
		ctx->dst_rect ? ctx->dst_rect : &dr, ctx->degree))
		printk(ctx->msg);

	if (ctx->wq) {
		rot_done = 1;
		wake_up(ctx->wq);
	}

	dmw96dma_free_list(ctx->list);
	kfree(ctx);
}

static void rotate_image(struct dmw96dma_image *src, struct dmw96dma_image *dst,
	struct dmw96dma_rect *src_rect, struct dmw96dma_rect *dst_rect,
	enum dmw96dma_rotate degree, char *msg, int wait)
{
	struct rot_ctx *ctx;
	DECLARE_WAIT_QUEUE_HEAD_ONSTACK(wq);
	int ret;

	ctx = kzalloc(sizeof(*ctx), GFP_KERNEL);
	if (!ctx)
		return;

	ctx->src = src;
	ctx->dst = dst;
	ctx->src_rect = src_rect;
	ctx->dst_rect = dst_rect;
	ctx->degree = degree;
	ctx->msg = msg;

	ctx->list = dmw96dma_alloc_image_list(rotate_image_finish, ctx);
	if (!ctx->list)
		goto free_ctx;

	ret = dmw96dma_add_rotate(ctx->list, src, dst, src_rect, dst_rect, degree, NULL, NULL);
	if (ret)
		goto free_list;

	if (wait) {
		ctx->wq = &wq;
		rot_done = 0;
	}

	ret = dmw96dma_submit(ctx->list);
	if (ret)
		goto free_list;

	if (wait) {
		wait_event_timeout(wq, rot_done, HZ);
		if (!rot_done) {
			printk("DMA rotate timed out!\n");
			goto free_list;
		}
	}

	return;

free_list:
	dmw96dma_free_list(ctx->list);
free_ctx:
	kfree(ctx);
}

static void test_rotate(int x, int y, enum dmw96dma_width fmt)
{
	struct dmw96dma_image *src, *dst_90_1, *dst_90_2, *dst_180;
	struct dmw96dma_rect sr, dr;
	struct dmw96dma_rect rect;

	printk("  Rotate %dx%d@%dbpp...", x, y, 8 << fmt);

	src = alloc_image(x, y, fmt, 0);
	dst_90_1 = alloc_image(y, x, fmt, 1);
	dst_90_2 = alloc_image(y, x, fmt, 0);
	dst_180 = alloc_image(x, y, fmt, 0);

	rotate_image(src, dst_90_1, NULL, NULL, DMW96DMA_ROTATE_90_CW, "90CW_FAIL ", 0);
	rotate_image(src, dst_90_2, NULL, NULL, DMW96DMA_ROTATE_90_CCW, "90CCW_FAIL ", 0);
	rotate_image(src, dst_180, NULL, NULL, DMW96DMA_ROTATE_180_CW, "180CW_FAIL ", 1);

	/* Test 90CW rotation with cropping. But need at least 4 tiles */
	if (x * (1 << fmt) < 4*64)
		goto out;
	if (x * (1 << fmt) < 4*64)
		goto out;

	/* reinitialize images */
	free_image(dst_90_1); dst_90_1 = alloc_image(y, x, fmt, 0);
	free_image(dst_90_2); dst_90_2 = alloc_image(y, x, fmt, 0);
	sr.x = 64 >> fmt; sr.y = 64 >> fmt; sr.width = 128 >> fmt; sr.height = 64 >> fmt;
	dr.x = 64 >> fmt; dr.y = 64 >> fmt;

	/* 90CW rotation with cropping */
	dr.width = sr.height;
	dr.height = sr.width;
	rotate_image(src, dst_90_1, &sr, &dr, DMW96DMA_ROTATE_90_CW, "90CW_WND_FAIL ", 1);

	/* Check that rest of destination image is unchanged */
	rect.x = rect.y = 0; rect.width = dst_90_1->width; rect.height = dr.y;
	if (cmp_image(dst_90_2, dst_90_1, &rect, &rect, -1))
		printk("90CW_WND_FAIL W1 ");

	rect.x = 0;
	rect.y = dr.y + dr.height;
	rect.width = dst_90_1->width;
	rect.height = dst_90_1->height - rect.y;
	if (cmp_image(dst_90_2, dst_90_1, &rect, &rect, -1))
		printk("90CW_WND_FAIL W2 ");

	rect.x = 0;
	rect.y = dr.y;
	rect.width = dr.x;
	rect.height = dr.height;
	if (cmp_image(dst_90_2, dst_90_1, &rect, &rect, -1))
		printk("90CW_WND_FAIL W3 ");

	rect.x = dr.x + dr.width;
	rect.y = dr.y;
	rect.width = dst_90_1->width - rect.x;
	rect.height = dr.height;
	if (cmp_image(dst_90_2, dst_90_1, &rect, &rect, -1))
		printk("90CW_WND_FAIL W4 ");

out:
	free_image(dst_180);
	free_image(dst_90_2);
	free_image(dst_90_1);
	free_image(src);

	printk("done\n");
}

static int rotate_test_init(void)
{
	printk("DMA rotate test...\n");

	test_rotate(16, 16, DMW96DMA_WIDTH_32);

	test_rotate(128, 128, DMW96DMA_WIDTH_32);
	test_rotate(128, 128, DMW96DMA_WIDTH_16);
	test_rotate(128, 128, DMW96DMA_WIDTH_8);

	test_rotate(480, 800, DMW96DMA_WIDTH_32);
	test_rotate(480, 800, DMW96DMA_WIDTH_16);

	test_rotate(512, 256, DMW96DMA_WIDTH_8);

	printk("DMA rotate test finished.\n");

	return 0;
}

late_initcall(rotate_test_init);
