/** \file
 * This is the network dependent layer to handle network related functionality.
 * This file is tightly coupled to neworking frame work of linux 2.6.xx kernel.
 * The functionality carried out in this file should be treated as an example only
 * if the underlying operating system is not Linux. 
 * 
 * \note Many of the functions other than the device specific functions
 *  changes for operating system other than Linux 2.6.xx
 * \internal 
 *-----------------------------REVISION HISTORY-----------------------------------
 * Synopsys			01/Aug/2007				Created
 */


//#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/init.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>


#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>

#include <linux/phy.h>
#include <linux/delay.h>

#include <linux/ccu.h>

#include "synopGMAC_Host.h"
#include "synopGMAC_plat.h"
#include "synopGMAC_network_interface.h"
#include "synopGMAC_Dev.h"
#include "synopGMAC_banner.h"

#define IPC_OFFLOAD

#define IOCTL_READ_REGISTER  SIOCDEVPRIVATE+1
#define IOCTL_WRITE_REGISTER SIOCDEVPRIVATE+2
#define IOCTL_READ_IPSTRUCT  SIOCDEVPRIVATE+3
#define IOCTL_READ_RXDESC    SIOCDEVPRIVATE+4
#define IOCTL_READ_TXDESC    SIOCDEVPRIVATE+5
#define IOCTL_POWER_DOWN     SIOCDEVPRIVATE+6

//static struct timer_list synopGMAC_cable_unplug_timer;
static u32 GMAC_Power_down; // This global variable is used to indicate the ISR whether the interrupts occured in the process of powering down the mac or not


/*These are the global pointers for their respecive structures*/
extern synopGMACNetworkAdapter * synopGMACadapter;
extern synopGMACdevice	          * synopGMACdev;
extern struct net_dev             * synopGMACnetdev;
extern struct platform_device             * synopGMACapbdev;

/*these are the global data for base address and its size*/
extern u8* synopGMACMappedAddr;
extern u32 synopGMACMappedAddrSize;
extern u32 synop_apb_using_dac;

/*Sample Wake-up frame filter configurations*/

u32 synopGMAC_wakeup_filter_config0[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x00000000,	// For Filter1 CRC is not computed may be it is 0x0000
					0x00000000,	// For Filter2 CRC is not computed may be it is 0x0000
					0x5F5F5F5F,     // For Filter3 CRC is based on 0,1,2,3,4,6,8,9,10,11,12,14,16,17,18,19,20,22,24,25,26,27,28,30 bytes from offset
					0x09000000,     // Filter 0,1,2 are disabled, Filter3 is enabled and filtering applies to only multicast packets
					0x1C000000,     // Filter 0,1,2 (no significance), filter 3 offset is 28 bytes from start of Destination MAC address 
					0x00000000,     // No significance of CRC for Filter0 and Filter1
					0xBDCC0000      // No significance of CRC for Filter2, Filter3 CRC is 0xBDCC
					};
u32 synopGMAC_wakeup_filter_config1[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x00000000,	// For Filter1 CRC is not computed may be it is 0x0000
					0x7A7A7A7A,	// For Filter2 CRC is based on 1,3,4,5,6,9,11,12,13,14,17,19,20,21,25,27,28,29,30 bytes from offset
					0x00000000,     // For Filter3 CRC is not computed may be it is 0x0000
					0x00010000,     // Filter 0,1,3 are disabled, Filter2 is enabled and filtering applies to only unicast packets
					0x00100000,     // Filter 0,1,3 (no significance), filter 2 offset is 16 bytes from start of Destination MAC address 
					0x00000000,     // No significance of CRC for Filter0 and Filter1
					0x0000A0FE      // No significance of CRC for Filter3, Filter2 CRC is 0xA0FE
					};
u32 synopGMAC_wakeup_filter_config2[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x000000FF,	// For Filter1 CRC is computed on 0,1,2,3,4,5,6,7 bytes from offset
					0x00000000,	// For Filter2 CRC is not computed may be it is 0x0000
					0x00000000,     // For Filter3 CRC is not computed may be it is 0x0000
					0x00000100,     // Filter 0,2,3 are disabled, Filter 1 is enabled and filtering applies to only unicast packets
					0x0000DF00,     // Filter 0,2,3 (no significance), filter 1 offset is 223 bytes from start of Destination MAC address 
					0xDB9E0000,     // No significance of CRC for Filter0, Filter1 CRC is 0xDB9E
					0x00000000      // No significance of CRC for Filter2 and Filter3 
					};

/*
The synopGMAC_wakeup_filter_config3[] is a sample configuration for wake up filter. 
Filter1 is used here
Filter1 offset is programmed to 50 (0x32)
Filter1 mask is set to 0x000000FF, indicating First 8 bytes are used by the filter
Filter1 CRC= 0x7EED this is the CRC computed on data 0x55 0x55 0x55 0x55 0x55 0x55 0x55 0x55

Refer accompanied software DWC_gmac_crc_example.c for CRC16 generation and how to use the same.
*/

u32 synopGMAC_wakeup_filter_config3[] = {
					0x00000000,	// For Filter0 CRC is not computed may be it is 0x0000
					0x000000FF,	// For Filter1 CRC is computed on 0,1,2,3,4,5,6,7 bytes from offset
					0x00000000,	// For Filter2 CRC is not computed may be it is 0x0000
					0x00000000,     // For Filter3 CRC is not computed may be it is 0x0000
					0x00000100,     // Filter 0,2,3 are disabled, Filter 1 is enabled and filtering applies to only unicast packets
					0x00003200,     // Filter 0,2,3 (no significance), filter 1 offset is 50 bytes from start of Destination MAC address 
					0x7eED0000,     // No significance of CRC for Filter0, Filter1 CRC is 0x7EED, 
					0x00000000      // No significance of CRC for Filter2 and Filter3 
					};

/**
 * Function used to detect the cable plugging and unplugging.
 * This function gets scheduled once in every second and polls
 * the PHY register for network cable plug/unplug. Once the 
 * connection is back the GMAC device is configured as per
 * new Duplex mode and Speed of the connection.
 * @param[in] u32 type but is not used currently. 
 * \return returns void.
 * \note This function is tightly coupled with Linux 2.6.xx.
 * \callgraph
 */
/*
static void synopGMAC_linux_cable_unplug_function(u32 notused)
{
	s32 status;
	u16 data;
	synopGMACNetworkAdapter *adapter = (synopGMACNetworkAdapter *)synopGMAC_cable_unplug_timer.data;
	synopGMACdevice            *gmacdev = adapter->synopGMACdev;

	init_timer(&synopGMAC_cable_unplug_timer);
	synopGMAC_cable_unplug_timer.expires = CHECK_TIME + jiffies;
	status = synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,PHY_Reg1, &data);

	if((data & PHY_Reg1_LINK_STATUS) == 0){
		TR1("No Link: %08x\n",data);
		gmacdev->LinkState = 0;
		gmacdev->DuplexMode = 0;
		gmacdev->Speed = 0;
		gmacdev->LoopBackMode = 0; 
	
		add_timer(&synopGMAC_cable_unplug_timer);
	}
	else{
		TR2("Link UP: %08x\n",data);
		if(!gmacdev->LinkState){
			status = synopGMAC_check_phy_init(gmacdev);
            synopGMAC_mac_init(gmacdev);
	
		}
		add_timer(&synopGMAC_cable_unplug_timer);
	}
	
}
*/
static void synopGMAC_linux_powerdown_mac(synopGMACdevice *gmacdev)
{
	TR2("Put the GMAC to power down mode..\n");
	// Disable the Dma engines in tx path
	GMAC_Power_down = 1;	// Let ISR know that Mac is going to be in the power down mode
	synopGMAC_disable_dma_tx(gmacdev);
	plat_delay(10000);		//allow any pending transmission to complete
	// Disable the Mac for both tx and rx
	synopGMAC_tx_disable(gmacdev);
	synopGMAC_rx_disable(gmacdev);
	plat_delay(10000); 		//Allow any pending buffer to be read by host
	//Disable the Dma in rx path
    synopGMAC_disable_dma_rx(gmacdev);

	//enable the power down mode
	//synopGMAC_pmt_unicast_enable(gmacdev);
	
	//prepare the gmac for magic packet reception and wake up frame reception
	synopGMAC_magic_packet_enable(gmacdev);
	synopGMAC_write_wakeup_frame_register(gmacdev, synopGMAC_wakeup_filter_config3);

	synopGMAC_wakeup_frame_enable(gmacdev);

	//gate the application and transmit clock inputs to the code. This is not done in this driver :).

	//enable the Mac for reception
    synopGMAC_rx_enable(gmacdev);

	//Enable the assertion of PMT interrupt
	synopGMAC_pmt_int_enable(gmacdev);
	//enter the power down mode
	synopGMAC_power_down_enable(gmacdev);
	return;
}

static void synopGMAC_linux_powerup_mac(synopGMACdevice *gmacdev)
{
	GMAC_Power_down = 0;	// Let ISR know that MAC is out of power down now
	if( synopGMAC_is_magic_packet_received(gmacdev))
		TR2("GMAC wokeup due to Magic Pkt Received\n");
	if(synopGMAC_is_wakeup_frame_received(gmacdev))
		TR2("GMAC wokeup due to Wakeup Frame Received\n");
	//Disable the assertion of PMT interrupt
	synopGMAC_pmt_int_disable(gmacdev);
	//Enable the mac and Dma rx and tx paths
    synopGMAC_rx_enable(gmacdev);
    synopGMAC_enable_dma_rx(gmacdev);

	synopGMAC_tx_enable(gmacdev);
	synopGMAC_enable_dma_tx(gmacdev);
	return;
}
/**
  * This sets up the transmit Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Device is interested only after the descriptors are setup. Therefore this function
  * is not included in the device driver API. This function should be treated as an
  * example code to design the descriptor structures for ring mode or chain mode.
  * This function depends on the allocation consistent dma-able
  * memory in case of linux. This limitation is due to the fact 
  *	- Allocates the memory for the descriptors.
  *	- Initialize the Busy and Next descriptors indices to 0(Indicating first descriptor).
  *	- Initialize the Busy and Next descriptors to first descriptor address.
  * 	- Initialize the last descriptor with the endof ring in case of ring mode.
  *	- Initialize the descriptors in chain mode.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to apb.
  * @param[in] number of descriptor expected in tx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note This function fails if allocation fails for required number of descriptors in Ring mode, but in chain mode
  *  function returns -ESYNOPGMACNOMEM in the process of descriptor chain creation. once returned from this function
  *  user should for gmacdev->TxDescCount to see how many descriptors are there in the chain. Should continue further
  *  only if the number of descriptors in the chain meets the requirements  
  */
s32 synopGMAC_setup_tx_desc_queue(synopGMACdevice * gmacdev,struct platform_device * apbdev,u32 no_of_desc, u32 desc_mode)
{
	s32 i;
	//u8* test = NULL;
	DmaDesc *first_desc = NULL;
	DmaDesc *second_desc = NULL;
	dma_addr_t dma_addr;
	gmacdev->TxDescCount = 0;
	
	
	if(desc_mode == RINGMODE){
		TR2("Total size of memory required for Tx Descriptors in Ring Mode = 0x%08x\n",((sizeof(DmaDesc) * no_of_desc)));
		first_desc = plat_alloc_consistent_dmaable_memory (apbdev, sizeof(DmaDesc) * no_of_desc,&dma_addr);
		if(first_desc == NULL){
			TR0("Error in Tx Descriptors memory allocation\n");
			return -ESYNOPGMACNOMEM;
		}
		gmacdev->TxDescCount = no_of_desc;
		gmacdev->TxDesc      = first_desc;
		gmacdev->TxDescDma   = dma_addr;
	
		TR2("gmacdev->TxDescDma = 0x%x \n",gmacdev->TxDescDma );
		
		for(i =0; i < gmacdev -> TxDescCount; i++){
			synopGMAC_tx_desc_init_ring(gmacdev->TxDesc + i, i == gmacdev->TxDescCount-1);
			TR2("%02d %08x \n",i, (unsigned int)(gmacdev->TxDesc + i) );
		}
	
	}
	else{
		//Allocate the head descriptor
		first_desc = plat_alloc_consistent_dmaable_memory (apbdev, sizeof(DmaDesc),&dma_addr);
		if(first_desc == NULL){
			TR0("Error in Tx Descriptor Memory allocation in Ring mode\n");
			return -ESYNOPGMACNOMEM;
		}
		gmacdev->TxDesc       = first_desc;
		gmacdev->TxDescDma    = dma_addr;
	
		TR2("Tx===================================================================Tx\n");
		first_desc->buffer2   = gmacdev->TxDescDma;
		first_desc->data2     = (u32)gmacdev->TxDesc;
	
		gmacdev->TxDescCount = 1;
		
		for(i =0; i <(no_of_desc-1); i++){
			second_desc = plat_alloc_consistent_dmaable_memory(apbdev, sizeof(DmaDesc),&dma_addr);
			if(second_desc == NULL){	
				TR0("Error in Tx Descriptor Memory allocation in Chain mode\n");
				return -ESYNOPGMACNOMEM;
			}
			first_desc->buffer2  = dma_addr;
			first_desc->data2    = (u32)second_desc;
			
			second_desc->buffer2 = gmacdev->TxDescDma;
			second_desc->data2   = (u32)gmacdev->TxDesc;
	
			synopGMAC_tx_desc_init_chain(first_desc);
			TR1("%02d %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->TxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
			gmacdev->TxDescCount += 1;
			first_desc = second_desc;
		}
			
			synopGMAC_tx_desc_init_chain(first_desc);
			TR1("%02d %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->TxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
		TR2("Tx===================================================================Tx\n");
	}

	gmacdev->TxNext = 0;
	gmacdev->TxBusy = 0;
	gmacdev->TxNextDesc = gmacdev->TxDesc;
	gmacdev->TxBusyDesc = gmacdev->TxDesc;
	gmacdev->BusyTxDesc  = 0; 

	return -ESYNOPGMACNOERR;
}


/**
  * This sets up the receive Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Device is interested only after the descriptors are setup. Therefore this function
  * is not included in the device driver API. This function should be treated as an
  * example code to design the descriptor structures in ring mode or chain mode.
  * This function depends on the pcidev structure for allocation of consistent dma-able memory in case of linux.
  * This limitation is due to the fact that linux uses pci structure to allocate a dmable memory
  *	- Allocates the memory for the descriptors.
  *	- Initialize the Busy and Next descriptors indices to 0(Indicating first descriptor).
  *	- Initialize the Busy and Next descriptors to first descriptor address.
  * 	- Initialize the last descriptor with the endof ring in case of ring mode.
  *	- Initialize the descriptors in chain mode.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to pci_device structure.
  * @param[in] number of descriptor expected in rx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note This function fails if allocation fails for required number of descriptors in Ring mode, but in chain mode
  *  function returns -ESYNOPGMACNOMEM in the process of descriptor chain creation. once returned from this function
  *  user should for gmacdev->RxDescCount to see how many descriptors are there in the chain. Should continue further
  *  only if the number of descriptors in the chain meets the requirements  
  */
s32 synopGMAC_setup_rx_desc_queue(synopGMACdevice * gmacdev,struct platform_device * apbdev,u32 no_of_desc, u32 desc_mode)
{
s32 i;

	DmaDesc *first_desc = NULL;
	DmaDesc *second_desc = NULL;
	dma_addr_t dma_addr;
	gmacdev->RxDescCount = 0;
	
	if(desc_mode == RINGMODE){
		TR2("total size of memory required for Rx Descriptors in Ring Mode = 0x%08x\n",((sizeof(DmaDesc) * no_of_desc)));
		first_desc = plat_alloc_consistent_dmaable_memory (apbdev, sizeof(DmaDesc) * no_of_desc, &dma_addr);
		if(first_desc == NULL){
			TR0("Error in Rx Descriptor Memory allocation in Ring mode\n");
			return -ESYNOPGMACNOMEM;
		}
		gmacdev->RxDescCount = no_of_desc;
		gmacdev->RxDesc      = first_desc;
		gmacdev->RxDescDma   = dma_addr;
	
		TR2("gmacdev->RxDescDma = 0x%x \n",gmacdev->RxDescDma );
	
		for(i =0; i < gmacdev -> RxDescCount; i++){
			synopGMAC_rx_desc_init_ring(gmacdev->RxDesc + i, i == gmacdev->RxDescCount-1);
			TR1("%02d %08x \n",i, (unsigned int)(gmacdev->RxDesc + i));
	
		}
	
	}
	else{
		//Allocate the head descriptor
		first_desc = plat_alloc_consistent_dmaable_memory (apbdev, sizeof(DmaDesc),&dma_addr);
		if(first_desc == NULL){
			TR0("Error in Rx Descriptor Memory allocation in Ring mode\n");
			return -ESYNOPGMACNOMEM;
		}
		gmacdev->RxDesc       = first_desc;
		gmacdev->RxDescDma    = dma_addr;
	
		TR2("Rx===================================================================Rx\n");
		first_desc->buffer2   = gmacdev->RxDescDma;
		first_desc->data2     = (u32) gmacdev->RxDesc;
	
		gmacdev->RxDescCount = 1;
		
		for(i =0; i < (no_of_desc-1); i++){
			second_desc = plat_alloc_consistent_dmaable_memory(apbdev, sizeof(DmaDesc),&dma_addr);
			if(second_desc == NULL){	
				TR0("Error in Rx Descriptor Memory allocation in Chain mode\n");
				return -ESYNOPGMACNOMEM;
			}
			first_desc->buffer2  = dma_addr;
			first_desc->data2    = (u32)second_desc;
			
			second_desc->buffer2 = gmacdev->RxDescDma;
			second_desc->data2   = (u32)gmacdev->RxDesc;
	
			synopGMAC_rx_desc_init_chain(first_desc);
			TR1("%02d  %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->RxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
			gmacdev->RxDescCount += 1;
			first_desc = second_desc;
		}
	
		synopGMAC_rx_desc_init_chain(first_desc);
		TR1("%02d  %08x %08x %08x %08x %08x %08x %08x \n",gmacdev->RxDescCount, (u32)first_desc, first_desc->status, first_desc->length, first_desc->buffer1,first_desc->buffer2, first_desc->data1, first_desc->data2);
		TR2("Rx===================================================================Rx\n");
	
	}
	
	gmacdev->RxNext = 0;
	gmacdev->RxBusy = 0;
	gmacdev->RxNextDesc = gmacdev->RxDesc;
	gmacdev->RxBusyDesc = gmacdev->RxDesc;
	
	gmacdev->BusyRxDesc   = 0; 
	
	return -ESYNOPGMACNOERR;
}

/**
  * This gives up the receive Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Once device's Dma is stopped the memory descriptor memory and the buffer memory deallocation,
  * is completely handled by the operating system, this call is kept outside the device driver Api.
  * This function should be treated as an example code to de-allocate the descriptor structures in ring mode or chain mode
  * and network buffer deallocation.
  * This function depends on the pcidev structure for dma-able memory deallocation for both descriptor memory and the
  * network buffer memory under linux.
  * The responsibility of this function is to 
  *     - Free the network buffer memory if any.
  *	- Fee the memory allocated for the descriptors.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to pci_device structure.
  * @param[in] number of descriptor expected in rx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note No referece should be made to descriptors once this function is called. This function is invoked when the device is closed.
  */
void synopGMAC_giveup_rx_desc_queue(synopGMACdevice * gmacdev, struct platform_device *apbdev, u32 desc_mode)
{
	s32 i;
	
	DmaDesc *first_desc = NULL;
	dma_addr_t first_desc_dma_addr;
	u32 status;
	dma_addr_t dma_addr1;
	dma_addr_t dma_addr2;
	u32 length1;
	u32 length2;
	u32 data1;
	u32 data2;
	
	if(desc_mode == RINGMODE){
		for(i =0; i < gmacdev -> RxDescCount; i++){
			synopGMAC_get_desc_data(gmacdev->RxDesc + i, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
			if((length1 != 0) && (data1 != 0)){
				dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr1 , 0 , 2);
				dev_kfree_skb((struct sk_buff *) data1);	// free buffer1
				TR2("(Ring mode) rx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
			}
			if((length2 != 0) && (data2 != 0)){
				dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr2 , 0 , 2);
				dev_kfree_skb((struct sk_buff *) data2);	//free buffer2
				TR2("(Ring mode) rx buffer2 %08x of size %d from %d rx descriptor is given back\n",data2, length2, i);
			}
		}
		plat_free_consistent_dmaable_memory(apbdev,(sizeof(DmaDesc) * gmacdev->RxDescCount),gmacdev->RxDesc,gmacdev->RxDescDma); //free descriptors memory
		TR2("Memory allocated %08x  for Rx Desriptors (ring) is given back\n",(u32)gmacdev->RxDesc);
	}
	else{
		TR2("rx-------------------------------------------------------------------rx\n");
		first_desc          = gmacdev->RxDesc;
		first_desc_dma_addr = gmacdev->RxDescDma;
		for(i =0; i < gmacdev -> RxDescCount; i++){
			synopGMAC_get_desc_data(first_desc, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
			TR1("%02d %08x %08x %08x %08x %08x %08x %08x\n",i,(u32)first_desc,first_desc->status,first_desc->length,first_desc->buffer1,first_desc->buffer2,first_desc->data1,first_desc->data2);
			if((length1 != 0) && (data1 != 0)){
				dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr1 , 0 , 2);
				dev_kfree_skb((struct sk_buff *) data1);	// free buffer1
				TR2("(Chain mode) rx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
			}
			plat_free_consistent_dmaable_memory(apbdev,(sizeof(DmaDesc)),first_desc,first_desc_dma_addr); //free descriptors
			TR2("Memory allocated %08x for Rx Descriptor (chain) at  %d is given back\n",data2,i);
	
			first_desc = (DmaDesc *)data2;
			first_desc_dma_addr = dma_addr2;
		}
	
		TR2("rx-------------------------------------------------------------------rx\n");
	}
	gmacdev->RxDesc    = NULL;
	gmacdev->RxDescDma = 0;
	return;
}

/**
  * This gives up the transmit Descriptor queue in ring or chain mode.
  * This function is tightly coupled to the platform and operating system
  * Once device's Dma is stopped the memory descriptor memory and the buffer memory deallocation,
  * is completely handled by the operating system, this call is kept outside the device driver Api.
  * This function should be treated as an example code to de-allocate the descriptor structures in ring mode or chain mode
  * and network buffer deallocation.
  * This function depends on the pcidev structure for dma-able memory deallocation for both descriptor memory and the
  * network buffer memory under linux.
  * The responsibility of this function is to 
  *     - Free the network buffer memory if any.
  *	- Fee the memory allocated for the descriptors.
  * @param[in] pointer to synopGMACdevice.
  * @param[in] pointer to pci_device structure.
  * @param[in] number of descriptor expected in tx descriptor queue.
  * @param[in] whether descriptors to be created in RING mode or CHAIN mode.
  * \return 0 upon success. Error code upon failure.
  * \note No reference should be made to descriptors once this function is called. This function is invoked when the device is closed.
  */
void synopGMAC_giveup_tx_desc_queue(synopGMACdevice * gmacdev,struct platform_device * apbdev, u32 desc_mode)
{
	s32 i;
	
	DmaDesc *first_desc = NULL;
	dma_addr_t first_desc_dma_addr;
	u32 status;
	dma_addr_t dma_addr1;
	dma_addr_t dma_addr2;
	u32 length1;
	u32 length2;
	u32 data1;
	u32 data2;
	
	if(desc_mode == RINGMODE){
		for(i =0; i < gmacdev -> TxDescCount; i++){
			synopGMAC_get_desc_data(gmacdev->TxDesc + i,&status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
			if((length1 != 0) && (data1 != 0)){
                dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr1 , 0 , 1);
				dev_kfree_skb((struct sk_buff *) data1);	// free buffer1
				TR2("(Ring mode) tx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
			}
			if((length2 != 0) && (data2 != 0)){
                dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr2 , 0 , 1);
				dev_kfree_skb((struct sk_buff *) data2);	//free buffer2
				TR2("(Ring mode) tx buffer2 %08x of size %d from %d rx descriptor is given back\n",data2, length2, i);
			}
		}
        plat_free_consistent_dmaable_memory(apbdev,(sizeof(DmaDesc) * gmacdev->TxDescCount),gmacdev->TxDesc,gmacdev->TxDescDma); //free descriptors
		TR1("Memory allocated %08x for Tx Desriptors (ring) is given back\n",(u32)gmacdev->TxDesc);
	}
	else{
		TR2("tx-------------------------------------------------------------------tx\n");
		first_desc          = gmacdev->TxDesc;
		first_desc_dma_addr = gmacdev->TxDescDma;
		for(i =0; i < gmacdev -> TxDescCount; i++){
			synopGMAC_get_desc_data(first_desc, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
			TR1("%02d %08x %08x %08x %08x %08x %08x %08x\n",i,(u32)first_desc,first_desc->status,first_desc->length,first_desc->buffer1,first_desc->buffer2,first_desc->data1,first_desc->data2);
			if((length1 != 0) && (data1 != 0)){
				dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr1 , 0 , 1);
				dev_kfree_skb((struct sk_buff *) data2);	// free buffer1
				TR2("(Chain mode) tx buffer1 %08x of size %d from %d rx descriptor is given back\n",data1, length1, i);
			}
			plat_free_consistent_dmaable_memory(apbdev,(sizeof(DmaDesc)),first_desc,first_desc_dma_addr); //free descriptors
			TR2("Memory allocated %08x for Tx Descriptor (chain) at  %d is given back\n",data2,i);
	
			first_desc = (DmaDesc *)data2;
			first_desc_dma_addr = dma_addr2;
		}
		TR1("tx-------------------------------------------------------------------tx\n");
	
	}
	gmacdev->TxDesc    = NULL;
	gmacdev->TxDescDma = 0;
	return;
}


/**
 * Function to handle housekeeping after a packet is transmitted over the wire.
 * After the transmission of a packet DMA generates corresponding interrupt 
 * (if it is enabled). It takes care of returning the sk_buff to the linux
 * kernel, updating the networking statistics and tracking the descriptors.
 * @param[in] pointer to net_device structure. 
 * \return void.
 * \note This function runs in interrupt context
 */
void synop_handle_transmit_over(struct net_device *netdev)
{
	synopGMACNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	struct platform_device *apbdev;
	s32 desc_index;
	u32 data1, data2;
	u32 status;
	u32 length1, length2;
	u32 dma_addr1, dma_addr2;
#ifdef ENH_DESC_8W
	u32 ext_status;
	u16 time_stamp_higher;
	u32 time_stamp_high;
	u32 time_stamp_low;
#endif
	adapter = netdev_priv(netdev);
	if(adapter == NULL){
		TR0("Unknown Device\n");
		return;
	}
	
	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL){
		TR0("GMAC device structure is missing\n");
		return;
	}
	apbdev  = (struct platform_device *)adapter->synopGMACapbdev;	
	/*Handle the transmit Descriptors*/
	do {
	#ifdef ENH_DESC_8W
		desc_index = synopGMAC_get_tx_qptr(gmacdev, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2,&ext_status,&time_stamp_high,&time_stamp_low);
			synopGMAC_TS_read_timestamp_higher_val(gmacdev, &time_stamp_higher);
	#else
		desc_index = synopGMAC_get_tx_qptr(gmacdev, &status, &dma_addr1, &length1, &data1, &dma_addr2, &length2, &data2);
	#endif
		//desc_index = synopGMAC_get_tx_qptr(gmacdev, &status, &dma_addr, &length, &data1);
		if(desc_index >= 0 && data1 != 0){
			TR2("Finished Transmit at Tx Descriptor %d for skb 0x%08x and buffer = %08x whose status is %08x \n", desc_index,data1,dma_addr1,status);
			#ifdef	IPC_OFFLOAD
			if(synopGMAC_is_tx_ipv4header_checksum_error(gmacdev, status)){
				TR0("Harware Failed to Insert IPV4 Header Checksum\n");
			}
			if(synopGMAC_is_tx_payload_checksum_error(gmacdev, status)){
				TR0("Harware Failed to Insert Payload Checksum\n");
			}
			#endif
		
			dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr1 , length1 , 1);
			dev_kfree_skb_irq((struct sk_buff *)data1);
	
			if(synopGMAC_is_desc_valid(status)){
				adapter->synopGMACNetStats.tx_bytes += length1;
				adapter->synopGMACNetStats.tx_packets++;
			}
			else {	
				TR0("Error in Status %08x\n",status);
				adapter->synopGMACNetStats.tx_errors++;
				adapter->synopGMACNetStats.tx_aborted_errors += synopGMAC_is_tx_aborted(status);
				adapter->synopGMACNetStats.tx_carrier_errors += synopGMAC_is_tx_carrier_error(status);
			}
		}
		adapter->synopGMACNetStats.collisions += synopGMAC_get_tx_collision_count(status);
	} while(desc_index >= 0);

	netif_wake_queue(netdev);
}




/**
 * Function to Receive a packet from the interface.
 * After Receiving a packet, DMA transfers the received packet to the system memory
 * and generates corresponding interrupt (if it is enabled). This function prepares
 * the sk_buff for received packet after removing the ethernet CRC, and hands it over
 * to linux networking stack.
 * 	- Updataes the networking interface statistics
 *	- Keeps track of the rx descriptors
 * @param[in] pointer to net_device structure. 
 * \return void.
 * \note This function runs in interrupt context.
 */

void synop_handle_received_data(struct net_device *netdev)
{
	synopGMACNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	struct platform_device *apbdev;

	s32 desc_index;
	
	u32 data1;
	u32 data2;
	u32 len;
	u32 status;
	u32 dma_addr1;
	u32 dma_addr2;
#ifdef ENH_DESC_8W
	u32 ext_status;
	u16 time_stamp_higher;
	u32 time_stamp_high;
	u32 time_stamp_low;
#endif
	//u32 length;
	
	struct sk_buff *skb; //This is the pointer to hold the received data
	
	adapter = netdev_priv(netdev);
	if(adapter == NULL){
		TR0("Unknown Device\n");
		return;
	}
	
	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL){
		TR0("GMAC device structure is missing\n");
		return;
	}

	apbdev  = (struct platform_device *)adapter->synopGMACapbdev;	
	/*Handle the Receive Descriptors*/
	do{
#ifdef ENH_DESC_8W
		desc_index = synopGMAC_get_rx_qptr(gmacdev, &status,&dma_addr1,NULL, &data1,&dma_addr2,NULL,&data2,&ext_status,&time_stamp_high,&time_stamp_low);
		if(desc_index >0){ 
			synopGMAC_TS_read_timestamp_higher_val(gmacdev, &time_stamp_higher);
			TR2("S:%08x ES:%08x DA1:%08x d1:%08x DA2:%08x d2:%08x TSH:%08x TSL:%08x TSHW:%08x \n",status,ext_status,dma_addr1, data1,dma_addr2,data2, time_stamp_high,time_stamp_low,time_stamp_higher);
		}
#else
		desc_index = synopGMAC_get_rx_qptr(gmacdev, &status,&dma_addr1,NULL, &data1,&dma_addr2,NULL,&data2);
#endif
		//desc_index = synopGMAC_get_rx_qptr(gmacdev, &status,&dma_addr,NULL, &data1);

		if(desc_index >= 0 && data1 != 0){
			TR2("Received Data at Rx Descriptor %d for skb 0x%08x whose status is %08x\n",desc_index,data1,status);
			/*At first step unmapped the dma address*/
			dma_unmap_single(apbdev == NULL ? NULL : &apbdev->dev, dma_addr1, 0, 2 );

			skb = (struct sk_buff *)data1;
			if(synopGMAC_is_rx_desc_valid(status)){
				len =  synopGMAC_get_rx_desc_frame_length(status) - 4; //Not interested in Ethernet CRC bytes
				skb_put(skb,len);
			#ifdef IPC_OFFLOAD
				// Now lets check for the IPC offloading
				/*  Since we have enabled the checksum offloading in hardware, lets inform the kernel
				    not to perform the checksum computation on the incoming packet. Note that ip header 
  				    checksum will be computed by the kernel immaterial of what we inform. Similary TCP/UDP/ICMP
				    pseudo header checksum will be computed by the stack. What we can inform is not to perform
				    payload checksum. 		
   				    When CHECKSUM_UNNECESSARY is set kernel bypasses the checksum computation.		    
				*/
	
				TR2("Checksum Offloading will be done now\n");
				skb->ip_summed = CHECKSUM_UNNECESSARY;
				
				#ifdef ENH_DESC_8W
				if(synopGMAC_is_ext_status(gmacdev, status)){ // extended status present indicates that the RDES4 need to be probed
					TR2("Extended Status present\n");
					if(synopGMAC_ES_is_IP_header_error(gmacdev,ext_status)){       // IP header (IPV4) checksum error
					//Linux Kernel doesnot care for ipv4 header checksum. So we will simply proceed by printing a warning ....
					TR0("(EXTSTS)Error in IP header error\n");
					skb->ip_summed = CHECKSUM_NONE;     //Let Kernel compute the checkssum
					}	
					if(synopGMAC_ES_is_rx_checksum_bypassed(gmacdev,ext_status)){   // Hardware engine bypassed the checksum computation/checking
					TR2("(EXTSTS)Hardware bypassed checksum computation\n");	
					skb->ip_summed = CHECKSUM_NONE;             // Let Kernel compute the checksum
					}
					if(synopGMAC_ES_is_IP_payload_error(gmacdev,ext_status)){       // IP payload checksum is in error (UDP/TCP/ICMP checksum error)
					TR0("(EXTSTS) Error in EP payload\n");	
					skb->ip_summed = CHECKSUM_NONE;             // Let Kernel compute the checksum
					}				
				}
				else{ // No extended status. So relevant information is available in the status itself
					if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxNoChkError ){
					TR1("Ip header and TCP/UDP payload checksum Bypassed <Chk Status = 4>  \n");
					skb->ip_summed = CHECKSUM_UNNECESSARY;	//Let Kernel bypass computing the Checksum
					}
					if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxIpHdrChkError ){
					//Linux Kernel doesnot care for ipv4 header checksum. So we will simply proceed by printing a warning ....
					TR0(" Error in 16bit IPV4 Header Checksum <Chk Status = 6>  \n");
					skb->ip_summed = CHECKSUM_UNNECESSARY;	//Let Kernel bypass the TCP/UDP checksum computation
					}				
					if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxLenLT600 ){
					TR1("IEEE 802.3 type frame with Length field Lesss than 0x0600 <Chk Status = 0> \n");
					skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
					}
					if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxIpHdrPayLoadChkBypass ){
					TR1("Ip header and TCP/UDP payload checksum Bypassed <Chk Status = 1>\n");
					skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
					}
					if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxChkBypass ){
					TR1("Ip header and TCP/UDP payload checksum Bypassed <Chk Status = 3>  \n");
					skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
					}
					if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxPayLoadChkError ){
					TR0(" TCP/UDP payload checksum Error <Chk Status = 5>  \n");
					skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
					}
					if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxIpHdrChkError ){
					//Linux Kernel doesnot care for ipv4 header checksum. So we will simply proceed by printing a warning ....
					TR0(" Both IP header and Payload Checksum Error <Chk Status = 7>  \n");
					skb->ip_summed = CHECKSUM_NONE;	        //Let Kernel compute the Checksum
					}
				}
				#else	
				if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxNoChkError ){
					TR2("Ip header and TCP/UDP payload checksum Bypassed <Chk Status = 4>  \n");
					skb->ip_summed = CHECKSUM_UNNECESSARY;	//Let Kernel bypass computing the Checksum
				}
				if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxIpHdrChkError ){
				//Linux Kernel doesnot care for ipv4 header checksum. So we will simply proceed by printing a warning ....
					TR0(" Error in 16bit IPV4 Header Checksum <Chk Status = 6>  \n");
					skb->ip_summed = CHECKSUM_UNNECESSARY;	//Let Kernel bypass the TCP/UDP checksum computation
				}				
				if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxLenLT600 ){
					TR2("IEEE 802.3 type frame with Length field Lesss than 0x0600 <Chk Status = 0> \n");
					skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
				}
				if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxIpHdrPayLoadChkBypass ){
					TR2("Ip header and TCP/UDP payload checksum Bypassed <Chk Status = 1>\n");
					skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
				}
				if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxChkBypass ){
					TR2("Ip header and TCP/UDP payload checksum Bypassed <Chk Status = 3>  \n");
				skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
				}
				if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxPayLoadChkError ){
					TR0(" TCP/UDP payload checksum Error <Chk Status = 5>  \n");
				skb->ip_summed = CHECKSUM_NONE;	//Let Kernel compute the Checksum
				}
				if(synopGMAC_is_rx_checksum_error(gmacdev, status) == RxIpHdrChkError ){
					//Linux Kernel doesnot care for ipv4 header checksum. So we will simply proceed by printing a warning ....
					TR0(" Both IP header and Payload Checksum Error <Chk Status = 7>  \n");
					skb->ip_summed = CHECKSUM_NONE;	        //Let Kernel compute the Checksum
				}
				
				#endif
			#endif //IPC_OFFLOAD	
				skb->dev = netdev;
				skb->protocol = eth_type_trans(skb, netdev);
				netif_rx(skb);

				netdev->last_rx = jiffies;
				adapter->synopGMACNetStats.rx_packets++;
				adapter->synopGMACNetStats.rx_bytes += len;
			}
			else{
				/*Now the present skb should be set free*/
				dev_kfree_skb_irq(skb);
				TR0("s: %08x\n",status);
				adapter->synopGMACNetStats.rx_errors++;
				adapter->synopGMACNetStats.collisions       += synopGMAC_is_rx_frame_collision(status);
				adapter->synopGMACNetStats.rx_crc_errors    += synopGMAC_is_rx_crc(status);
				adapter->synopGMACNetStats.rx_frame_errors  += synopGMAC_is_frame_dribbling_errors(status);
				adapter->synopGMACNetStats.rx_length_errors += synopGMAC_is_rx_frame_length_errors(status);
			}
			
			//Now lets allocate the skb for the emptied descriptor
			skb = dev_alloc_skb(netdev->mtu + ETHERNET_PACKET_EXTRA);
			if(skb == NULL){
				TR0("SKB memory allocation failed \n");
				adapter->synopGMACNetStats.rx_dropped++;
			}
						
			dma_addr1 = dma_map_single(apbdev == NULL ? NULL : &apbdev->dev, skb->data,skb_tailroom(skb),2);
			
			desc_index = synopGMAC_set_rx_qptr(gmacdev,dma_addr1, skb_tailroom(skb), (u32)skb,0,0,0);
			
			if(desc_index < 0){
				TR0("Cannot set Rx Descriptor for skb %08x\n",(u32)skb);
				dev_kfree_skb_irq(skb);
			}
					
		}
	}while(desc_index >= 0);
}


/**
 * Interrupt service routing.
 * This is the function registered as ISR for device interrupts.
 * @param[in] interrupt number. 
 * @param[in] void pointer to device unique structure (Required for shared interrupts in Linux).
 * @param[in] pointer to pt_regs (not used).
 * \return Returns IRQ_NONE if not device interrupts IRQ_HANDLED for device interrupts.
 * \note This function runs in interrupt context
 *
 */
irqreturn_t synopGMAC_intr_handler(s32 intr_num, void * dev_id)
{       
	/*Kernels passes the netdev structure in the dev_id. So grab it*/
	struct net_device *netdev;
	synopGMACNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	struct platform_device *apbdev;
	u32 interrupt,dma_status_reg;
	s32 status;
	u32 dma_addr;

	netdev  = (struct net_device *) dev_id;
	if(netdev == NULL){
			TR0("Unknown Device\n");
			return -1;
	}
	
	adapter  = netdev_priv(netdev);
	if(adapter == NULL){
			TR0("Adapter Structure Missing\n");
			return -1;
	}
	
	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL){
			TR0("GMAC device structure Missing\n");
			return -1;
	}

	apbdev  = (struct platform_device *)adapter->synopGMACapbdev;	

	/*Read the Dma interrupt status to know whether the interrupt got generated by our device or not*/
	dma_status_reg = synopGMACReadReg((u32 *)gmacdev->DmaBase, DmaStatus);
	
	if(dma_status_reg == 0)
		return IRQ_NONE;

    synopGMAC_disable_interrupt_all(gmacdev);

    TR2("Dma Status Reg: 0x%08x\n",dma_status_reg);
	
	if(dma_status_reg & GmacPmtIntr){
		TR2(": Interrupt due to PMT module\n");
		synopGMAC_linux_powerup_mac(gmacdev);
	}
	
	if(dma_status_reg & GmacMmcIntr){
		TR2(": Interrupt due to MMC module\n");
		TR2(": synopGMAC_rx_int_status = %08x\n",synopGMAC_read_mmc_rx_int_status(gmacdev));
		TR2(": synopGMAC_tx_int_status = %08x\n",synopGMAC_read_mmc_tx_int_status(gmacdev));
	}

	if(dma_status_reg & GmacLineIntfIntr){
		TR2(": Interrupt due to GMAC LINE module\n");
	}

	/*Now lets handle the DMA interrupts
	 this is also clear the interrupts ! ! !*/ 
    interrupt = synopGMAC_get_interrupt_type(gmacdev);

	/*
	 * ATTENTION: The following call will guarantee that all memory
	 * transactions indicated by the device irq flags have actually hit the
	 * memory. Just process the flags which have been sampled. When reading
	 * the status again any new IRQ flags MUST NOT be processed until
	 * ccu_barrier() is called again!
	 */
	ccu_barrier();

	TR2(":Interrupts to be handled: 0x%08x\n",interrupt);

	if(interrupt & synopGMACDmaError){



		TR0(":Fatal Bus Error Inetrrupt Seen\n");

		synopGMAC_disable_dma_tx(gmacdev);
		synopGMAC_disable_dma_rx(gmacdev);
				
		synopGMAC_take_desc_ownership_tx(gmacdev);
		synopGMAC_take_desc_ownership_rx(gmacdev);
		
		synopGMAC_init_tx_rx_desc_queue(gmacdev);
		
		synopGMAC_reset(gmacdev);//reset the DMA engine and the GMAC ip

		synopGMAC_set_mac_addr(gmacdev, GmacAddr0High,GmacAddr0Low, gmacdev->mac_addr);
		synopGMAC_dma_bus_mode_init(gmacdev,DmaFixedBurstEnable| DmaBurstLength8 | DmaDescriptorSkip2 );
		synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward);	
		synopGMAC_init_rx_desc_base(gmacdev);
		synopGMAC_init_tx_desc_base(gmacdev);
		synopGMAC_mac_init(gmacdev);
        synopGMAC_enable_dma_rx(gmacdev);
		synopGMAC_enable_dma_tx(gmacdev);
		TR0("\n");
	}


	if(interrupt & synopGMACDmaRxNormal){
		TR2(": Rx Normal \n");
                synop_handle_received_data(netdev);
	}

		if(interrupt & synopGMACDmaRxAbnormal){
			TR1(":Abnormal Rx Interrupt Seen\n");
		#if 1
		
		if(GMAC_Power_down == 0){	// If Mac is not in powerdown
			adapter->synopGMACNetStats.rx_over_errors++;
			/*Now Descriptors have been created in synop_handle_received_data(). Just issue a poll demand to resume DMA operation*/
			synopGMAC_resume_dma_rx(gmacdev);//To handle GBPS with 12 descriptors
		}
		#endif
	}

	if(interrupt & synopGMACDmaRxStopped){
        TR1(":Receiver stopped seeing Rx interrupts\n"); //Receiver gone in to stopped state
		#if 1
		if(GMAC_Power_down == 0){	// If Mac is not in powerdown
			adapter->synopGMACNetStats.rx_over_errors++;
			do{
				struct sk_buff *skb = alloc_skb(netdev->mtu + ETHERNET_HEADER + ETHERNET_CRC, GFP_ATOMIC);
				if(skb == NULL){
					TR0(":ERROR in skb buffer allocation Better Luck Next time\n");
					break;
					//			return -ESYNOPGMACNOMEM;
				}
				
                dma_addr = dma_map_single(apbdev == NULL ? NULL : &apbdev->dev, skb->data,skb_tailroom(skb),2);
			
				status = synopGMAC_set_rx_qptr(gmacdev,dma_addr, skb_tailroom(skb), (u32)skb,0,0,0);
				TR1(":Set Rx Descriptor no %08x for skb %08x \n",status,(u32)skb);
				if(status < 0)
					dev_kfree_skb_irq(skb);//changed from dev_free_skb. If problem check this again--manju
			
			}while(status >= 0);

            synopGMAC_enable_dma_rx(gmacdev);
		}
		#endif
	}

	if(interrupt & synopGMACDmaTxNormal){
		//xmit function has done its job
		TR2(":Finished Normal Transmission \n");
		synop_handle_transmit_over(netdev);//Do whatever you want after the transmission is over
	}

	if(interrupt & synopGMACDmaTxAbnormal){
		TR0(":Abnormal Tx Interrupt Seen\n");
		#if 1
		if(GMAC_Power_down == 0){	// If Mac is not in powerdown
			synop_handle_transmit_over(netdev);
		}
		#endif
	}



	if(interrupt & synopGMACDmaTxStopped){
		TR2(":Transmitter stopped sending the packets\n");
		#if 1
	    if(GMAC_Power_down == 0){	// If Mac is not in powerdown
			synopGMAC_disable_dma_tx(gmacdev);
			synopGMAC_take_desc_ownership_tx(gmacdev);
            synopGMAC_enable_dma_tx(gmacdev);
			netif_wake_queue(netdev);
			TR2(":Transmission Resumed\n");
		}
		#endif
	}

    /* Enable the interrrupt before returning from ISR*/
	synopGMAC_enable_interrupt(gmacdev,DmaIntEnable);
	return IRQ_HANDLED;
}


static int macg_mdio_read(struct mii_bus *bus, int mii_id, int regnum)
{

	synopGMACdevice*  gmacdev = bus->priv;
	unsigned short data = -1;
	int status = synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase, mii_id , regnum , &data);

	if(status != -ESYNOPGMACNOERR)
		TR0("ERROR in Phy read\n");

	return data;

}

static int macg_mdio_write(struct mii_bus *bus, int mii_id, int regnum, unsigned short value)
{
    synopGMACdevice*  gmacdev = bus->priv;
	int status = synopGMAC_write_phy_reg((u32 *)gmacdev->MacBase, mii_id ,regnum,value);

	if(status != -ESYNOPGMACNOERR)
		TR0("ERROR in Phy write\n");

	return 0;

}

static int macg_mdio_reset(struct mii_bus *bus)
{
	return 0;
}

static void macb_handle_link_change(struct net_device *dev)
{
	synopGMACNetworkAdapter* adapter = netdev_priv(dev);
	synopGMACdevice* gmacdev = (synopGMACdevice *)adapter->synopGMACdev;
	struct phy_device *phydev = adapter->phy_dev;
	unsigned long flags;
	int status_change = 0;
	
	spin_lock_irqsave(&adapter->lock, flags);

	if (phydev->link) {
		if ((gmacdev->Speed != phydev->speed) || (gmacdev->DuplexMode != phydev->duplex)) {
			gmacdev->Speed = phydev->speed;
			gmacdev->DuplexMode = phydev->duplex;
			synopGMAC_mac_init(gmacdev);
			status_change = 1;
		}
	}

	if (phydev->link != gmacdev->LinkState) {
		if (!phydev->link) {
			gmacdev->Speed = 0;
			gmacdev->DuplexMode = -1;
		}
		gmacdev->LinkState = phydev->link;
		synopGMAC_mac_init(gmacdev);
		status_change = 1;
	}

	spin_unlock_irqrestore(&adapter->lock, flags);

	if (status_change) {
		if (phydev->link)
			printk(KERN_INFO "%s: link up (%d/%s)\n",
			       dev->name, phydev->speed,
			       DUPLEX_FULL == phydev->duplex ? "Full":"Half");
		else
			printk(KERN_INFO "%s: link down\n", dev->name);
	}

}


static int macg_mii_probe(struct net_device *dev)
{
	synopGMACNetworkAdapter* adapter = (synopGMACNetworkAdapter *) netdev_priv(dev);
	synopGMACdevice* gmacdev = (synopGMACdevice *)adapter->synopGMACdev;
	struct phy_device *phydev = NULL;
	int phy_addr;

	TR2("called \n");

	/* find the first phy */
	for (phy_addr = 0; phy_addr < PHY_MAX_ADDR; phy_addr++) {
		if (adapter->mii_bus->phy_map[phy_addr]) {
			phydev = adapter->mii_bus->phy_map[phy_addr];
			break;
		}
	}

	if (!phydev) {
		printk (KERN_ERR "%s: no PHY found\n", dev->name);
		return -1;
	}

	/* attach the mac to the phy */
    phydev = phy_connect(dev,  dev_name(&phydev->dev)  /*phydev->dev.bus_id*/, &macb_handle_link_change, 0, PHY_INTERFACE_MODE_MII);
    
	if (IS_ERR(phydev)) {
		printk(KERN_ERR "%s: Could not attach to PHY\n", dev->name);
		return PTR_ERR(phydev);
	}

	/* mask with MAC supported features */
	phydev->supported &= PHY_BASIC_FEATURES;
    
	phydev->advertising = phydev->supported;

	gmacdev->LinkState = 0;		/* Link status as reported by the Marvel Phy                */
	gmacdev->DuplexMode = -1;               /* Duplex mode of the Phy				    */
	gmacdev->Speed = 0;			/* Speed of the Phy					    */
	gmacdev->LoopBackMode = 0; 		/* Loopback status of the Phy				    */
	adapter->phy_dev = phydev;

	return 0;
}

/*
static int macg_poll(struct napi_struct *napi, int budget)
{
	//TODO please complete
	return 0;
}
*/

/**
 * Function used when the interface is opened for use.
 * We register synopGMAC_linux_open function to linux open(). Basically this 
 * function prepares the the device for operation . This function is called whenever ifconfig (in Linux)
 * activates the device (for example "ifconfig eth0 up"). This function registers
 * system resources needed 
 * 	- Attaches device to device specific structure
 * 	- Programs the MDC clock for PHY configuration
 * 	- Check and initialize the PHY interface 
 *	- ISR registration
 * 	- Setup and initialize Tx and Rx descriptors
 *	- Initialize MAC and DMA
 *	- Allocate Memory for RX descriptors (The should be DMAable)
 * 	- Initialize one second timer to detect cable plug/unplug
 *	- Configure and Enable Interrupts
 *	- Enable Tx and Rx
 *	- start the Linux network queue interface
 * @param[in] pointer to net_device structure. 
 * \return Returns 0 on success and error status upon failure.
 * \callgraph
 */

#define MACG_DEV_ID 1234

static s32 synopGMAC_linux_open(struct net_device *netdev)
{
	s32 status = 0;
	s32 retval = 0;
	s32 ijk;
	//s32 reserve_len=2;
	u32 dma_addr;
	struct sk_buff *skb;
	synopGMACNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	struct platform_device *apbdev;

	TR2("called \n");
	adapter = (synopGMACNetworkAdapter *) netdev_priv(netdev);
	gmacdev = (synopGMACdevice *)adapter->synopGMACdev;
	apbdev  = (struct platform_device *)adapter->synopGMACapbdev;	
	
	/*Now platform dependent initialization.*/

	/*Lets reset the IP*/
    TR2("adapter= %08x gmacdev = %08x netdev = %08x apbdev= %08x\n",(u32)adapter,(u32)gmacdev,(u32)netdev,(u32)apbdev);
	synopGMAC_reset(gmacdev);

	
	/*Attach the device to MAC struct This will configure all the required base addresses
	  such as Mac base, configuration base, phy base address(out of 32 possible phys )*/
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	synopGMAC_attach(synopGMACadapter->synopGMACdev,(u32) synopGMACMappedAddr + MACBASE,(u32) synopGMACMappedAddr + DMABASE, DEFAULT_PHY_UNDEFINE);
#else
	synopGMAC_attach(synopGMACadapter->synopGMACdev,(u32) synopGMACMappedAddr + MACBASE,(u32) synopGMACMappedAddr + DMABASE, DEFAULT_PHY_BASE);
#endif

	/*Lets read the version of ip in to device structure*/	
	synopGMAC_read_version(gmacdev);

	synopGMAC_get_mac_addr(synopGMACadapter->synopGMACdev,GmacAddr0High,GmacAddr0Low, netdev->dev_addr); 
    
	//Now set the broadcast address
	for(ijk = 0; ijk <6; ijk++){
		netdev->broadcast[ijk] = 0xff;
	}

	for(ijk = 0; ijk <6; ijk++){
		TR2("netdev->dev_addr[%d] = %02x and netdev->broadcast[%d] = %02x\n",ijk,netdev->dev_addr[ijk],ijk,netdev->broadcast[ijk]);
	}

    /*Check for Phy initialization*/
	synopGMAC_set_mdc_clk_div(gmacdev,GmiiCsrClk2);
	gmacdev->ClockDivMdc = synopGMAC_get_mdc_clk_div(gmacdev);

	adapter->mii_bus = mdiobus_alloc();
	if (adapter->mii_bus == NULL) {
		retval = -ENOMEM;
		goto gmac_open_err_out;
	}

	adapter->mii_bus->name = "macg_mii_bus";
	adapter->mii_bus->read = &macg_mdio_read;
	adapter->mii_bus->write = &macg_mdio_write;
	adapter->mii_bus->reset = &macg_mdio_reset;
	snprintf(adapter->mii_bus->id, MII_BUS_ID_SIZE, "%x", MACG_DEV_ID);
	adapter->mii_bus->priv = gmacdev;
	adapter->mii_bus->parent = &netdev->dev;
	//TODO complete NAPI
	//netif_napi_add(netdev, &adapter->napi, macg_poll, 64);
	
	platform_set_drvdata(apbdev, adapter->mii_bus);

	if (mdiobus_register(adapter->mii_bus))
		goto gmac_err_out_free_mdiobus;

	if (macg_mii_probe(netdev) != 0) {
		goto gmac_err_out_unregister_bus;
	}

	/*Set up the tx and rx descriptor queue/ring*/
	synopGMAC_setup_tx_desc_queue(gmacdev,apbdev,TRANSMIT_DESC_SIZE, RINGMODE);
//	synopGMAC_setup_tx_desc_queue(gmacdev,apbdev,TRANSMIT_DESC_SIZE, CHAINMODE);

	synopGMAC_init_tx_desc_base(gmacdev);	//Program the transmit descriptor base address in to DmaTxBase addr

	synopGMAC_setup_rx_desc_queue(gmacdev,apbdev,RECEIVE_DESC_SIZE, RINGMODE);
//	synopGMAC_setup_rx_desc_queue(gmacdev,apbdev,RECEIVE_DESC_SIZE, CHAINMODE);

	synopGMAC_init_rx_desc_base(gmacdev);	//Program the transmit descriptor base address in to DmaTxBase addr

//	synopGMAC_dma_bus_mode_init(gmacdev,DmaFixedBurstEnable| DmaBurstLength8 | DmaDescriptorSkip2 );
//	synopGMAC_dma_bus_mode_init(gmacdev, DmaBurstLength32 | DmaDescriptorSkip2 );
//	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaTxSecondFrame);	
	
//	synopGMAC_dma_bus_mode_init(gmacdev,DmaFixedBurstEnable | DmaBurstLength8 | DmaDescriptorSkip2 ); //pbl8 incrx
//	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaTxSecondFrame|DmaRxThreshCtrl64);	

//	synopGMAC_dma_bus_mode_init(gmacdev, DmaBurstLength8 | DmaDescriptorSkip2 );                      //pbl8 incr
//	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaTxSecondFrame|DmaRxThreshCtrl64);	

//	synopGMAC_dma_bus_mode_init(gmacdev,DmaFixedBurstEnable | DmaBurstLength16 | DmaDescriptorSkip2 ); //pbl16 incrx
//	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaTxSecondFrame|DmaRxThreshCtrl64);	

//	synopGMAC_dma_bus_mode_init(gmacdev, DmaBurstLength16 | DmaDescriptorSkip2 );                      //pbl16 incr
//	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaTxSecondFrame|DmaRxThreshCtrl64);	

#ifdef ENH_DESC_8W
	synopGMAC_dma_bus_mode_init(gmacdev, DmaBurstLength32 | DmaDescriptorSkip2 | DmaDescriptor8Words ); //pbl32 incr with rxthreshold 128 and Desc is 8 Words
#else
	synopGMAC_dma_bus_mode_init(gmacdev, DmaBurstLength32 | DmaDescriptorSkip2);                      //pbl32 incr with rxthreshold 128
#endif
	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaTxSecondFrame|DmaRxThreshCtrl128);	

//	synopGMAC_dma_bus_mode_init(gmacdev, DmaBurstLength64 | DmaDescriptorSkip2 );                      //pbl64 incr with rxthreshold 128
//	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaTxSecondFrame|DmaRxThreshCtrl128);	

//	synopGMAC_dma_bus_mode_init(gmacdev, DmaBurstLength64 | DmaDescriptorSkip2 );                      //pbl64 incr with rxthreshold 128
//	synopGMAC_dma_control_init(gmacdev,DmaStoreAndForward |DmaRxThreshCtrl128);	

	/*Initialize the mac interface*/

	synopGMAC_mac_init(gmacdev);
	synopGMAC_pause_control(gmacdev); // This enables the pause control in Full duplex mode of operation
	#ifdef IPC_OFFLOAD
	/*IPC Checksum offloading is enabled for this driver. Should only be used if Full Ip checksumm offload engine is configured in the hardware*/
	synopGMAC_enable_rx_chksum_offload(gmacdev);  	//Enable the offload engine in the receive path
	synopGMAC_rx_tcpip_chksum_drop_enable(gmacdev); // This is default configuration, DMA drops the packets if error in encapsulated ethernet payload
							// The FEF bit in DMA control register is configured to 0 indicating DMA to drop the errored frames.
	/*Inform the Linux Networking stack about the hardware capability of checksum offloading*/
	netdev->features = NETIF_F_HW_CSUM;
	#endif

     do{
		skb = alloc_skb(netdev->mtu + ETHERNET_HEADER + ETHERNET_CRC, GFP_ATOMIC);
		if(skb == NULL){
			TR0("ERROR in skb buffer allocation\n");
			break;
//			return -ESYNOPGMACNOMEM;
		}
//		skb_reserve(skb,reserve_len);
//		TR2("skb = %08x skb->tail = %08x skb_tailroom(skb)=%08x skb->data = %08x\n",(u32)skb,(u32)skb->tail,(skb_tailroom(skb)),(u32)skb->data);
//		skb->dev = netdev;
		dma_addr = dma_map_single(apbdev == NULL ? NULL : &apbdev->dev, skb->data,skb_tailroom(skb), 2);
		status = synopGMAC_set_rx_qptr(gmacdev,dma_addr, skb_tailroom(skb), (u32)skb,0,0,0);
		if(status < 0)
			dev_kfree_skb(skb);
			
	}while(status >= 0);
	/*
	TR2("Setting up the cable unplug timer\n");
	init_timer(&synopGMAC_cable_unplug_timer);
	synopGMAC_cable_unplug_timer.function = (void *)synopGMAC_linux_cable_unplug_function;
	synopGMAC_cable_unplug_timer.data = (u32) adapter;
	synopGMAC_cable_unplug_timer.expires = CHECK_TIME + jiffies;
	add_timer(&synopGMAC_cable_unplug_timer);
	*/
	synopGMAC_clear_interrupt(gmacdev);
	/*
	Disable the interrupts generated by MMC and IPC counters.
	If these are not disabled ISR should be modified accordingly to handle these interrupts.
	*/	
	synopGMAC_disable_mmc_tx_interrupt(gmacdev, 0xFFFFFFFF);
	synopGMAC_disable_mmc_rx_interrupt(gmacdev, 0xFFFFFFFF);
	synopGMAC_disable_mmc_ipc_rx_interrupt(gmacdev, 0xFFFFFFFF);

	synopGMAC_enable_interrupt(gmacdev,DmaIntEnable);
    synopGMAC_enable_dma_rx(gmacdev);
	synopGMAC_enable_dma_tx(gmacdev);
	phy_start(adapter->phy_dev);
	netif_start_queue(netdev);
	return 0;

gmac_err_out_unregister_bus:
	mdiobus_unregister(adapter->mii_bus);
gmac_err_out_free_mdiobus:
	mdiobus_free(adapter->mii_bus);
gmac_open_err_out:
	return retval;

}

/**
 * Function used when the interface is closed.
 *
 * This function is registered to linux stop() function. This function is 
 * called whenever ifconfig (in Linux) closes the device (for example "ifconfig eth0 down").
 * This releases all the system resources allocated during open call.
 * system resources int needs 
 * 	- Disable the device interrupts
 * 	- Stop the receiver and get back all the rx descriptors from the DMA
 * 	- Stop the transmitter and get back all the tx descriptors from the DMA 
 * 	- Stop the Linux network queue interface
 *	- Free the irq (ISR registered is removed from the kernel)
 * 	- Release the TX and RX descripor memory
 *	- De-initialize one second timer rgistered for cable plug/unplug tracking
 * @param[in] pointer to net_device structure. 
 * \return Returns 0 on success and error status upon failure.
 * \callgraph
 */

static s32 synopGMAC_linux_close(struct net_device *netdev)
{
	
//	s32 status = 0;
//	s32 retval = 0;
//	u32 dma_addr;
	synopGMACNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
    struct platform_device *apbdev;
	
	TR2("%s\n",__FUNCTION__);

	if (netdev == NULL) {

		TR0("OOPS network device is null\n");
		return -1;
	}

	adapter = (synopGMACNetworkAdapter *) netdev_priv(netdev);
	if(adapter == NULL){
		TR0("OOPS adapter is null\n");
		return -1;
	}

	gmacdev = (synopGMACdevice *) adapter->synopGMACdev;
	if(gmacdev == NULL){
		TR0("OOPS gmacdev is null\n");
		return -1;
	}

	apbdev = (struct platform_device *)adapter->synopGMACapbdev;
    if(apbdev == NULL){
		TR0("OOPS apbdev is null\n");
		return -1;
	}

	/*Disable all the interrupts*/
	synopGMAC_disable_interrupt_all(gmacdev);
	TR2("the synopGMAC interrupt has been disabled\n");

	/*Disable the reception*/	
	synopGMAC_disable_dma_rx(gmacdev);
    synopGMAC_take_desc_ownership_rx(gmacdev);
	TR2("the synopGMAC Reception has been disabled\n");

	/*Disable the transmission*/
	synopGMAC_disable_dma_tx(gmacdev);
        synopGMAC_take_desc_ownership_tx(gmacdev);

	TR2("the synopGMAC Transmission has been disabled\n");
	netif_stop_queue(netdev);
	/*Now free the irq: This will detach the interrupt handler registered*/
	//free_irq(pcidev->irq, netdev);
	//free_irq(platform_get_irq(apbdev, 0), netdev);
	//TR2("the synopGMAC interrupt handler has been removed\n");
	
	/*Free the Rx Descriptor contents*/
	TR2("Now calling synopGMAC_giveup_rx_desc_queue \n");
	synopGMAC_giveup_rx_desc_queue(gmacdev, apbdev, RINGMODE);
//	synopGMAC_giveup_rx_desc_queue(gmacdev, apbdev, CHAINMODE);
	TR2("Now calling synopGMAC_giveup_tx_desc_queue \n");
	synopGMAC_giveup_tx_desc_queue(gmacdev, apbdev, RINGMODE);
//	synopGMAC_giveup_tx_desc_queue(gmacdev, apbdev, CHAINMODE);
	
	TR2("Freeing the cable unplug timer\n");	
	//del_timer(&synopGMAC_cable_unplug_timer);
	if (adapter->phy_dev)
		phy_stop(adapter->phy_dev);

	if (adapter->phy_dev)
		phy_disconnect(adapter->phy_dev);

	mdiobus_unregister(adapter->mii_bus);
	mdiobus_free(adapter->mii_bus);
    return -ESYNOPGMACNOERR;

}

/**
 * Function to transmit a given packet on the wire.
 * Whenever Linux Kernel has a packet ready to be transmitted, this function is called.
 * The function prepares a packet and prepares the descriptor and 
 * enables/resumes the transmission.
 * @param[in] pointer to sk_buff structure. 
 * @param[in] pointer to net_device structure.
 * \return Returns 0 on success and Error code on failure. 
 * \note structure sk_buff is used to hold packet in Linux networking stacks.
 */
static s32 synopGMAC_linux_xmit_frames(struct sk_buff *skb, struct net_device *netdev)
{
	s32 status = 0;
	s32 counter =0;
	u32 offload_needed = 0;
	u32 dma_addr;
	//u32 flags;
	synopGMACNetworkAdapter *adapter;
	synopGMACdevice * gmacdev;
	unsigned int i = 0;
	struct platform_device *apbdev;
	TR2("called \n");
	if(skb == NULL){
		TR0("skb is NULL What happened to Linux Kernel? \n ");
		return -1;
	}
	
	adapter = (synopGMACNetworkAdapter *) netdev_priv(netdev);
	if(adapter == NULL)
		return -1;

	gmacdev = (synopGMACdevice *) adapter->synopGMACdev;
	if(gmacdev == NULL)
		return -1;

	apbdev  = (struct platform_device *)adapter->synopGMACapbdev;	
	/*Stop the network queue*/	
	netif_stop_queue(netdev); 

		
	//if(skb->ip_summed == CHECKSUM_HW){
	if(skb->ip_summed == CHECKSUM_PARTIAL){
		/*	
		   In Linux networking, if kernel indicates skb->ip_summed = CHECKSUM_HW, then only checksum offloading should be performed
		   Make sure that the OS on which this code runs have proper support to enable offloading.
		*/
		offload_needed = 0x00000001;
		//#if 0
		//printk(KERN_ERR"skb->ip_summed = CHECKSUM_HW\n");
		//printk(KERN_ERR"skb->h.th=%08x skb->h.th->check=%08x\n",(u32)(skb->h.th),(u32)(skb->h.th->check));
		//printk(KERNs_CRIT"skb->h.uh=%08x skb->h.uh->check=%08x\n",(u32)(skb->h.uh),(u32)(skb->h.uh->check));
		//printk(KERN_ERR"\n skb->len = %d skb->mac_len = %d skb->data = %08x skb->csum = %08x skb->h.raw = %08x\n",skb->len,skb->mac_len,(u32)(skb->data),skb->csum,(u32)(skb->h.raw));
		//printk(KERN_ERR"DST MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+0),*(skb->data+1),*(skb->data+2),*(skb->data+3),*(skb->data+4),*(skb->data+5));
		//printk(KERN_ERR"SRC MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+6),*(skb->data+7),*(skb->data+8),*(skb->data+9),*(skb->data+10),*(skb->data+11));
		//printk(KERN_ERR"Len/type    :%02x %02x\n",*(skb->data+12),*(skb->data+13));
		if(((*(skb->data+14)) & 0xF0) == 0x40){
			//printk(KERN_ERR"IPV4 Header:\n");
			//printk(KERN_ERR"%02x %02x %02x %02x\n",*(skb->data+14),*(skb->data+15),*(skb->data+16),*(skb->data+17));
			//printk(KERN_ERR"%02x %02x %02x %02x\n",*(skb->data+18),*(skb->data+19),*(skb->data+20),*(skb->data+21));
			//printk(KERN_ERR"%02x %02x %02x %02x\n",*(skb->data+22),*(skb->data+23),*(skb->data+24),*(skb->data+25));
			//printk(KERN_ERR"%02x %02x %02x %02x\n",*(skb->data+26),*(skb->data+27),*(skb->data+28),*(skb->data+29));
			//printk(KERN_ERR"%02x %02x %02x %02x\n\n",*(skb->data+30),*(skb->data+31),*(skb->data+32),*(skb->data+33));
			for(counter = 34; counter < skb->len; counter++) {
					//printk("%02X ",*(skb->data + counter));
			}
		}
		else{
			printk(KERN_ERR"IPV6 FRAME:\n");
			for(counter = 14; counter < skb->len; counter++)
				printk("%02X ",*(skb->data + counter));
		}
		//#endif
	}

	//printk(KERN_ERR"DST MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+0),*(skb->data+1),*(skb->data+2),*(skb->data+3),*(skb->data+4),*(skb->data+5));
	//printk(KERN_ERR"SRC MAC addr:%02x %02x %02x %02x %02x %02x\n",*(skb->data+6),*(skb->data+7),*(skb->data+8),*(skb->data+9),*(skb->data+10),*(skb->data+11));

	/*Now we have skb ready and OS invoked this function. Lets make our DMA know about this*/
	dma_addr = dma_map_single(apbdev == NULL ? NULL : &apbdev->dev, skb->data,skb->len, 1);
	TR2("dma_addr = 0x%x ,skb->data = 0x%p , skb->len = 0x%x \n", dma_addr , skb->data , skb->len);
    for (i = 0 ; i < skb->len ; i++)
	{
		TR2("skb->data[%d] = 0x%x\n",i , skb->data[i]);
	}
    status = synopGMAC_set_tx_qptr(gmacdev, dma_addr, skb->len, (u32)skb,0,0,0,offload_needed);
	if(status < 0){
		TR0("%s No More Free Tx Descriptors\n",__FUNCTION__);
//		dev_kfree_skb (skb); //with this, system used to freeze.. ??
		return -EBUSY;
	}
	
	/*Now force the DMA to start transmission*/	
	synopGMAC_resume_dma_tx(gmacdev);
	netdev->trans_start = jiffies;
	
	/*Now start the netdev queue*/
	netif_wake_queue(netdev);
	
	return -ESYNOPGMACNOERR;
}

/**
 * Function provides the network interface statistics.
 * Function is registered to linux get_stats() function. This function is 
 * called whenever ifconfig (in Linux) asks for networkig statistics
 * (for example "ifconfig eth0").
 * @param[in] pointer to net_device structure. 
 * \return Returns pointer to net_device_stats structure.
 * \callgraph
 */
static struct net_device_stats *  synopGMAC_linux_get_stats(struct net_device *netdev)
{
	TR2("called \n");
	return( &(((synopGMACNetworkAdapter *)(netdev_priv(netdev)))->synopGMACNetStats) );
}

/**
 * Function to set multicast and promiscous mode.
 * @param[in] pointer to net_device structure. 
 * \return returns void.
 */
static void synopGMAC_linux_set_multicast_list(struct net_device *netdev)
{
	TR2("called \n");
//todo Function not yet implemented.
return;
}

/**
 * Function to set ethernet address of the NIC.
 * @param[in] pointer to net_device structure. 
 * @param[in] pointer to an address structure. 
 * \return Returns 0 on success Errorcode on failure.
 */
static s32 synopGMAC_linux_set_mac_address(struct net_device *netdev, void * macaddr)
{

	synopGMACNetworkAdapter *adapter = NULL;
	synopGMACdevice * gmacdev = NULL;
	struct sockaddr *addr = macaddr;

	adapter = (synopGMACNetworkAdapter *) netdev_priv(netdev);
	if(adapter == NULL)
		return -1;

	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL)
		return -1;
	
	if(!is_valid_ether_addr(addr->sa_data))
		return -EADDRNOTAVAIL;
	
	synopGMAC_set_mac_addr(gmacdev,GmacAddr0High,GmacAddr0Low, addr->sa_data); 
	synopGMAC_get_mac_addr(synopGMACadapter->synopGMACdev,GmacAddr0High,GmacAddr0Low, netdev->dev_addr); 
	
	TR2("%s called \n",__FUNCTION__);
	return 0;
}

/**
 * Function to change the Maximum Transfer Unit.
 * @param[in] pointer to net_device structure. 
 * @param[in] New value for maximum frame size.
 * \return Returns 0 on success Errorcode on failure.
 */
static s32 synopGMAC_linux_change_mtu(struct net_device *netdev, s32 newmtu)
{
	TR2("%s called \n",__FUNCTION__);
	//todo Function not yet implemented.
	return 0;

}

/**
 * IOCTL interface.
 * This function is mainly for debugging purpose.
 * This provides hooks for Register read write, Retrieve descriptor status
 * and Retreiving Device structure information.
 * @param[in] pointer to net_device structure. 
 * @param[in] pointer to ifreq structure.
 * @param[in] ioctl command. 
 * \return Returns 0 on success Error code on failure.
 */
static s32 synopGMAC_linux_do_ioctl(struct net_device *netdev, struct ifreq *ifr, s32 cmd)
{
	s32 retval = 0;
	u16 temp_data = 0;
	synopGMACNetworkAdapter *adapter = NULL;
	synopGMACdevice * gmacdev = NULL;
	struct ifr_data_struct
	{
		u32 unit;
		u32 addr;
		u32 data;
	} *req;

	
	if(netdev == NULL)
		return -1;
	if(ifr == NULL)
		return -1;
	
	req = (struct ifr_data_struct *)ifr->ifr_data;
	
	adapter = (synopGMACNetworkAdapter *) netdev_priv(netdev);
	if(adapter == NULL)
		return -1;
	
	gmacdev = adapter->synopGMACdev;
	if(gmacdev == NULL)
		return -1;
	TR2("%s :: on device %s req->unit = %08x req->addr = %08x req->data = %08x cmd = %08x \n",__FUNCTION__,netdev->name,req->unit,req->addr,req->data,cmd);
	
	switch(cmd)
	{
		case IOCTL_READ_REGISTER:		//IOCTL for reading IP registers : Read Registers
			if      (req->unit == 0)	// Read Mac Register
				req->data = synopGMACReadReg((u32 *)gmacdev->MacBase,req->addr);
			else if (req->unit == 1)	// Read DMA Register
				req->data = synopGMACReadReg((u32 *)gmacdev->DmaBase,req->addr);
			else if (req->unit == 2){	// Read Phy Register
				temp_data = phy_read(adapter->phy_dev, req->addr);// synopGMAC_read_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,req->addr,&temp_data);
				req->data = (u32)temp_data;
				//if(retval != -ESYNOPGMACNOERR)
				//	TR0("ERROR in Phy read\n");	
			}
			break;
	
		case IOCTL_WRITE_REGISTER:		//IOCTL for reading IP registers : Read Registers
			if      (req->unit == 0)	// Write Mac Register
				synopGMACWriteReg((u32 *)gmacdev->MacBase,req->addr,req->data);
			else if (req->unit == 1)	// Write DMA Register
				synopGMACWriteReg((u32 *)gmacdev->DmaBase,req->addr,req->data);
			else if (req->unit == 2){	// Write Phy Register
				retval = phy_write(adapter->phy_dev , req->addr,req->data); //synopGMAC_write_phy_reg((u32 *)gmacdev->MacBase,gmacdev->PhyBase,req->addr,req->data);
				//if(retval != -ESYNOPGMACNOERR)
				//	TR0("ERROR in Phy read\n");	
			}
			break;
	
		case IOCTL_READ_IPSTRUCT:		//IOCTL for reading GMAC DEVICE IP private structure
				memcpy(ifr->ifr_data, gmacdev, sizeof(synopGMACdevice));
			break;
	
		case IOCTL_READ_RXDESC:			//IOCTL for Reading Rx DMA DESCRIPTOR
			memcpy(ifr->ifr_data, gmacdev->RxDesc + ((DmaDesc *) (ifr->ifr_data))->data1, sizeof(DmaDesc) );
			break;
	
		case IOCTL_READ_TXDESC:			//IOCTL for Reading Tx DMA DESCRIPTOR
			memcpy(ifr->ifr_data, gmacdev->TxDesc + ((DmaDesc *) (ifr->ifr_data))->data1, sizeof(DmaDesc) );
			break;
		case IOCTL_POWER_DOWN:
			if	(req->unit == 1){	//power down the mac
				TR2("============I will Power down the MAC now =============\n");
				// If it is already in power down don't power down again
				retval = 0;
				if(((synopGMACReadReg((u32 *)gmacdev->MacBase,GmacPmtCtrlStatus)) & GmacPmtPowerDown) != GmacPmtPowerDown){
				synopGMAC_linux_powerdown_mac(gmacdev);			
				retval = 0;
				}
			}
			if	(req->unit == 2){	//Disable the power down  and wake up the Mac locally
				TR2("============I will Power up the MAC now =============\n");
				//If already powered down then only try to wake up
				retval = -1;
				if(((synopGMACReadReg((u32 *)gmacdev->MacBase,GmacPmtCtrlStatus)) & GmacPmtPowerDown) == GmacPmtPowerDown){
				synopGMAC_power_down_disable(gmacdev);
				synopGMAC_linux_powerup_mac(gmacdev);
				retval = 0;
				}
			}
			break;
		default:
			retval = -1;
	
	}


	return retval;
}

/**
 * Function to handle a Tx Hang.
 * This is a software hook (Linux) to handle transmitter hang if any.
 * We get transmitter hang in the device interrupt status, and is handled
 * in ISR. This function is here as a place holder.
 * @param[in] pointer to net_device structure 
 * \return void.
 */
static void synopGMAC_linux_tx_timeout(struct net_device *netdev)
{
	TR2("%s called \n",__FUNCTION__);
	//todo Function not yet implemented
	return;
}



static int macg_get_settings(struct net_device *dev, struct ethtool_cmd *cmd)
{
	
	synopGMACNetworkAdapter *adapter = netdev_priv(dev);
	struct phy_device *phydev = adapter->phy_dev;

	if (!phydev)
		return -ENODEV;

	return phy_ethtool_gset(phydev, cmd);
	
	return 0;
}

static int macg_set_settings(struct net_device *dev, struct ethtool_cmd *cmd)
{
	
	synopGMACNetworkAdapter *adapter = netdev_priv(dev);
	struct phy_device *phydev = adapter->phy_dev;

	if (!phydev)
		return -ENODEV;

	return phy_ethtool_sset(phydev, cmd);

	return 0;
}

static void macg_get_drvinfo(struct net_device *dev,struct ethtool_drvinfo *info)
{
    strcpy(info->driver, synopGMAC_driver_string);
	strcpy(info->version, synopGMAC_driver_version);
	strcpy(info->bus_info, "None");
}


static struct ethtool_ops macg_ethtool_ops = {
	.get_settings		= macg_get_settings,
	.set_settings		= macg_set_settings,
	.get_drvinfo		= macg_get_drvinfo,
	.get_link		= ethtool_op_get_link,
};

static void dmw96macg_clk_enable(void)
{
}

static const struct net_device_ops netdev_ops = {
	.ndo_open		= synopGMAC_linux_open,
	.ndo_stop		= synopGMAC_linux_close,
	.ndo_start_xmit		= synopGMAC_linux_xmit_frames,
	.ndo_set_multicast_list	= synopGMAC_linux_set_multicast_list,
	.ndo_set_mac_address	= synopGMAC_linux_set_mac_address,
	.ndo_do_ioctl		= synopGMAC_linux_do_ioctl,
	.ndo_change_mtu		= synopGMAC_linux_change_mtu,
	.ndo_tx_timeout		= synopGMAC_linux_tx_timeout,
	.ndo_get_stats		= synopGMAC_linux_get_stats,
};

/**
 * Function to initialize the Linux network interface.
 * 
 * Linux dependent Network interface is setup here. This provides 
 * an example to handle the network dependent functionality.
 *
 * \return Returns 0 on success and Error code on failure.
 */
s32 __init synopGMAC_init_network_interface(void)
{
	s32 err = 0;
	struct net_device *netdev;
	
	TR2("Now Going to Call register_netdev to register the network interface for GMAC core\n");
	/*
	Lets allocate and set up an ethernet device, it takes the sizeof the private structure. This is mandatory as a 32 byte 
	allignment is required for the private data structure.
	*/
	netdev = alloc_etherdev(sizeof(synopGMACNetworkAdapter));
	if(!netdev){
	err = -ESYNOPGMACNOMEM;
	goto err_alloc_etherdev;
	}
	
	synopGMACadapter = netdev_priv(netdev);
	synopGMACadapter->synopGMACnetdev = netdev;
	synopGMACadapter->synopGMACapbdev = synopGMACapbdev;
	synopGMACadapter->synopGMACdev    = NULL;

	
	/*Allocate Memory for the the GMACip structure*/
	synopGMACadapter->synopGMACdev = (synopGMACdevice *) plat_alloc_memory(sizeof (synopGMACdevice));
	if(!synopGMACadapter->synopGMACdev){
		TR0("Error in Memory Allocataion \n");
	}

	synopGMACadapter->clk = clk_get(&synopGMACapbdev->dev, "dmw96_gmac");
	if (IS_ERR(synopGMACadapter->clk)) {
		err = PTR_ERR(synopGMACadapter->clk);
		TR0("Cannot get clock\n");
		goto err_alloc_irq;
	}
	clk_enable(synopGMACadapter->clk);

	spin_lock_init(&synopGMACadapter->lock);

	/*Attach the device to MAC struct This will configure all the required base addresses
	  such as Mac base, configuration base, phy base address(out of 32 possible phys )*/
	synopGMAC_attach(synopGMACadapter->synopGMACdev,(u32) synopGMACMappedAddr + MACBASE,(u32) synopGMACMappedAddr + DMABASE, DEFAULT_PHY_BASE);
	synopGMAC_save_mac_addr(synopGMACadapter->synopGMACdev);
	synopGMAC_reset(synopGMACadapter->synopGMACdev);

    if(synop_apb_using_dac){
		TR2("netdev->features = %08x\n",(unsigned int)netdev->features);
		TR2("synop_apb_using dac is %08x\n",synop_apb_using_dac);
		netdev->features |= NETIF_F_HIGHDMA ;
		TR2("netdev->features = %08x\n",(unsigned int)netdev->features);
	}

	netdev->netdev_ops = &netdev_ops;
	netdev->watchdog_timeo = 5 * HZ;
	netdev->ethtool_ops = &macg_ethtool_ops;
	//random_ether_addr(netdev->dev_addr);
	synopGMAC_get_mac_addr(synopGMACadapter->synopGMACdev,GmacAddr0High,GmacAddr0Low, netdev->dev_addr); 

	dmw96macg_clk_enable();

	/*Request for an shared interrupt*/
	netdev->irq = platform_get_irq(synopGMACadapter->synopGMACapbdev, 0);
	if(request_irq (platform_get_irq(synopGMACadapter->synopGMACapbdev, 0), synopGMAC_intr_handler, 0 /*IRQF_SHARED*/ , netdev->name,netdev)){
 		TR0("Error in request_irq\n");
		goto err_alloc_irq;	
	}

    /*Now start the network interface*/
	TR2("Now Registering the netdevice\n");
	err = register_netdev(netdev);
	if(err) {
		TR0("Error in Registering netdevice\n");
		goto err_register_netdev;
	}  

	TR2("%s owns a interrupt on line %d\n",netdev->name, platform_get_irq(synopGMACadapter->synopGMACapbdev, 0));

 	return 0;

err_register_netdev:
	free_irq(platform_get_irq(synopGMACadapter->synopGMACapbdev, 0), netdev);
err_alloc_irq:
	plat_free_memory(synopGMACadapter->synopGMACdev);
err_alloc_etherdev:

return err;
}


/**
 * Function to initialize the Linux network interface.
 * Linux dependent Network interface is setup here. This provides 
 * an example to handle the network dependent functionality.
 * \return Returns 0 on success and Error code on failure.
 */
void __exit synopGMAC_exit_network_interface(void)
{
	clk_disable(synopGMACadapter->clk);
	clk_put(synopGMACadapter->clk);

	TR2("Now Calling network_unregister\n");
	unregister_netdev(synopGMACadapter->synopGMACnetdev);

	/*Now free the irq: This will detach the interrupt handler registered*/
    free_irq(platform_get_irq(synopGMACadapter->synopGMACapbdev, 0), synopGMACadapter->synopGMACnetdev);
	TR2("the synopGMAC interrupt handler has been removed\n");

}



